<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use App\Services\WhmcsService;

class DnsController extends Controller
{
    protected $whmcs;
    protected $cpanel;

    public function __construct(WhmcsService $whmcs, cPanelController $cpanel)
    {
        $this->whmcs = $whmcs;
        $this->cpanel = $cpanel;
    }

    public function getZone(Request $request, string $domain)
    {
        $clientId = Session::get('client');

        // Hash kontrolü
        $expected = hash_hmac('sha256', $domain, $clientId . env('DOMAIN_HASH_SECRET'));
        if (!hash_equals($expected, (string)$request->input('hash'))) {
            abort(403, 'Yetkisiz erişim!');
        }

        $serverId = (int) $request->input('serverId');

        $res = $this->cpanel->getZoneFromWhm($serverId, $domain);

        if (!($res['ok'] ?? false)) {
            // Hata durumunda JSON yerine hata sayfası veya uygun mesaj dönmeli
             return response()->view('errors.custom', ['message' => $res['error'] ?? 'Zone alınamadı'], 500);
        }

        return view('dns', [
            'dnsZone'     => $res['records'],
            'domain'      => $domain,
            'serverId'    => $serverId,
            'hash'        => $request->input('hash')
        ]);
    }

    public function updateRecord(Request $request, string $domain)
    {
        $clientId = Session::get('client');

        // Hash kontrolü
        $expected = hash_hmac('sha256', $domain, $clientId . env('DOMAIN_HASH_SECRET'));
        if (!hash_equals($expected, (string)$request->input('hash'))) {
             return response()->json(['status' => 'error', 'message' => 'Yetkisiz erişim (Hash Hatası)'], 403);
        }

        $serverId = (int) $request->input('serverId');

        // Blade tarafından { records: [...] } formatında gönderiliyor
        $records = $request->input('records');
        $payload = $records[0] ?? []; // Tek kayıt güncellemesi için ilk elemanı al

        if (empty($payload)) {
             return response()->json(['status' => 'error', 'message' => 'Kayıt verisi bulunamadı.'], 400);
        }

        // Line varsa güncelle, yoksa ekle
        $line = (int)($payload['line'] ?? 0);

        if ($line > 0) {
            $res = $this->cpanel->updateZoneRecordFromWhm($serverId, $domain, $payload);
        } else {
            $res = $this->cpanel->addZoneRecordFromWhm($serverId, $domain, $payload);
        }

        if (!($res['ok'] ?? false)) {
            return response()->json(['status' => 'error', 'message' => $res['error'] ?? 'İşlem başarısız', 'raw' => $res['raw'] ?? null], 500);
        }

        return response()->json(['status' => 'success']);
    }

    public function deleteRecord(Request $request, string $domain)
    {
        $clientId = Session::get('client');

        // Hash kontrolü
        $expected = hash_hmac('sha256', $domain, $clientId . env('DOMAIN_HASH_SECRET'));
        if (!hash_equals($expected, (string)$request->input('hash'))) {
            return response()->json(['status' => 'error', 'message' => 'Yetkisiz erişim (Hash Hatası)'], 403);
        }

        $serverId = (int) $request->input('serverId');
        $line = (int) $request->input('line');

        if ($line <= 0) {
            return response()->json(['status' => 'error', 'message' => 'Line numarası eksik.'], 400);
        }

        $res = $this->cpanel->deleteZoneRecordFromWhm($serverId, $domain, $line);

        if (!($res['ok'] ?? false)) {
            return response()->json(['status' => 'error', 'message' => $res['error'] ?? 'Silme işlemi başarısız', 'raw' => $res['raw'] ?? null], 500);
        }

        return response()->json(['status' => 'success']);
    }
}
