<?php

namespace App\Filament\Resources;

use App\Filament\Resources\CarrierResource\Pages;
use App\Filament\Resources\CarrierResource\RelationManagers;
use App\Models\Carrier;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Resources\Pages\Page;
use Filament\Pages\SubNavigationPosition;
use Filament\Support\Enums\FontWeight;
use Filament\Support\Enums\IconPosition;
use Filament\Support\Enums\MaxWidth;

use Filament\Infolists;
use Filament\Infolists\Infolist;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class CarrierResource extends Resource
{
    protected static ?string $model = Carrier::class;

    protected static ?string $recordTitleAttribute = 'name';

    protected static ?string $modelLabel = 'Tedarikçi';
    protected static ?string $pluralModelLabel = 'Tedarikçiler';

    // protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $navigationLabel = 'Tedarikçiler';
    protected static ?string $navigationGroup = 'Uygulama';

    protected static ?int $navigationSort = 2;

    protected static int $globalSearchResultsLimit = 5;

    protected static SubNavigationPosition $subNavigationPosition = SubNavigationPosition::Top;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('Tedarikçi Bilgileri')
                    ->columns([
                        'lg' => 2,
                        'xl' => 4,
                    ])
                    ->icon('heroicon-m-briefcase')
                    ->schema([
                        Forms\Components\TextInput::make('name')
                            ->label('Tedarikçi Adı')
                            ->maxLength(255)
                            ->required(),

                        Forms\Components\TextInput::make('tax_number')
                            ->label('Vergi Numarası')
                            ->maxLength(255),

                        Forms\Components\TextInput::make('tax_office')
                            ->label('Vergi Dairesi')
                            ->maxLength(255),

                        Forms\Components\TextInput::make('address')
                            ->label('Adres')
                            ->maxLength(255)
                            ->columnSpan(2),

                        Forms\Components\TextInput::make('telefon')
                            ->label('Telefon')
                            ->maxLength(255),

                        Forms\Components\TextInput::make('email')
                            ->label('E-posta Adresi')
                            ->maxLength(150),

                        Forms\Components\TextInput::make('website')
                            ->label('Website')
                            ->prefix('https://')
                            ->maxLength(255),
                    ]),

                Forms\Components\Section::make('Tedarikçi Bağlantıları')
                    ->columns([
                        'lg' => 1,
                    ])
                    ->icon('heroicon-m-users')
                    ->schema([
                        Forms\Components\Repeater::make('contacts')
                            ->label('')
                            ->relationship()
                            ->schema([
                                Forms\Components\TextInput::make('name')
                                    ->label('Kişi')
                                    ->required()
                                    ->maxLength(100),

                                Forms\Components\TextInput::make('phone_number')
                                    ->label('Telefon')
                                    ->maxLength(100),

                                Forms\Components\TextInput::make('email')
                                    ->label('E-posta Adresi')
                                    ->email()
                                    ->maxLength(150),

                                Forms\Components\Textarea::make('notes')
                                    ->label('Notlar')
                                    ->autosize()
                                    ->columnSpanFull(),
                            ])
                            ->itemLabel(fn (array $state): ?string => $state['name'] ?? null)
                            ->addActionLabel('Yeni Kişi Ekle')
                            ->orderColumn('sort_order')
                            ->reorderable(true)
                            ->reorderableWithButtons()
                            ->collapsible()
                            ->collapsed()
                            ->deleteAction(fn (Forms\Components\Actions\Action $action) => $action->requiresConfirmation())
                            ->defaultItems(0)
                            ->columns(3),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->defaultSort('id', 'desc')
            // ->heading(static::$pluralModelLabel.' Listesi')
            ->recordUrl(null)
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->label('Tedarikçi')
                    ->weight(FontWeight::Medium)
                    ->description(fn ($record) => $record?->website)
                    ->searchable(),

                Tables\Columns\TextColumn::make('tax_number')
                    ->label('Vergi Numarası')
                    ->searchable(),

                Tables\Columns\TextColumn::make('tax_office')
                    ->label('Vergi Dairesi'),

                Tables\Columns\TextColumn::make('telefon')
                    ->label('Telefon')
                    ->searchable(),

                Tables\Columns\TextColumn::make('email')
                    ->label('E-posta Adresi')
                    ->searchable(),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('Oluşturma')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                Tables\Columns\TextColumn::make('updated_at')
                    ->label('Güncelleme')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Tables\Actions\ViewAction::make(),
                    Tables\Actions\EditAction::make(),
                    Tables\Actions\DeleteAction::make(),
                ]),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function infolist(Infolist $infolist): Infolist
    {
        return $infolist
            ->schema([
                Infolists\Components\Tabs::make('Tabs')
                    ->tabs([
                        Infolists\Components\Tabs\Tab::make('Tedarikçi Bilgileri')
                            ->schema([
                                Infolists\Components\TextEntry::make('name')
                                    ->label('Tedarikçi Adı:')
                                    ->inlineLabel(),

                                Infolists\Components\TextEntry::make('tax_number')
                                    ->label('Vergi Numarası:')
                                    ->inlineLabel()
                                    ->default('-'),

                                Infolists\Components\TextEntry::make('tax_office')
                                    ->label('Vergi Dairesi:')
                                    ->inlineLabel()
                                    ->default('-'),

                                Infolists\Components\TextEntry::make('telefon')
                                    ->label('Telefon:')
                                    ->inlineLabel()
                                    ->default('-'),

                                Infolists\Components\TextEntry::make('email')
                                    ->label('E-posta Adresi:')
                                    ->icon('heroicon-c-envelope')
                                    ->iconPosition(IconPosition::After)
                                    ->hidden(fn ($state): bool => !$state)
                                    ->url(fn ($state): string => 'mailto:'.$state)
                                    ->inlineLabel()
                                    ->default('-'),

                                Infolists\Components\TextEntry::make('website')
                                    ->label('Website:')
                                    ->icon('heroicon-c-arrow-top-right-on-square')
                                    ->iconPosition(IconPosition::After)
                                    ->url(fn ($state): string => 'https://'.$state)
                                    ->openUrlInNewTab()
                                    ->inlineLabel()
                                    ->default('-'),
                            ]),

                        Infolists\Components\Tabs\Tab::make('Hesap Numaraları')
                            ->schema([
                                Infolists\Components\RepeatableEntry::make('bankAccounts')
                                    ->label('')
                                    ->schema([
                                        Infolists\Components\TextEntry::make('name')
                                            ->label('Banka:')
                                            ->inlineLabel(),

                                        Infolists\Components\TextEntry::make('recipient')
                                            ->label('Alıcı:')
                                            ->inlineLabel()
                                            ->default('-'),

                                        Infolists\Components\TextEntry::make('iban')
                                            ->label('IBAN:')
                                            ->icon('heroicon-o-document-duplicate')
                                            ->iconPosition(IconPosition::After)
                                            ->copyable()
                                            ->copyMessageDuration(1100)
                                            ->inlineLabel(),

                                        Infolists\Components\TextEntry::make('account_number')
                                            ->label('Hesap Numarası:')
                                            ->inlineLabel()
                                            ->default('-'),

                                        Infolists\Components\TextEntry::make('branch_name')
                                            ->label('Şube:')
                                            ->inlineLabel()
                                            ->default('-'),
                                    ]),
                            ]),

                        Infolists\Components\Tabs\Tab::make('Bağlantılar')
                            ->schema([
                                Infolists\Components\RepeatableEntry::make('contacts')
                                    ->label('')
                                    ->schema([
                                        Infolists\Components\TextEntry::make('name')
                                            ->label('Kişi:')
                                            ->inlineLabel(),

                                        Infolists\Components\TextEntry::make('phone_number')
                                            ->label('Telefon:')
                                            ->hidden(fn ($state): bool => !$state)
                                            ->inlineLabel(),

                                        Infolists\Components\TextEntry::make('email')
                                            ->label('E-posta Adresi:')
                                            ->icon('heroicon-c-envelope')
                                            ->iconPosition(IconPosition::After)
                                            ->hidden(fn ($state): bool => !$state)
                                            ->url(fn ($state): string => 'mailto:'.$state)
                                            ->inlineLabel(),

                                        Infolists\Components\TextEntry::make('notes')
                                            ->label('Notlar:')
                                            ->hidden(fn ($state): bool => !$state)
                                            ->inlineLabel(),
                                    ]),
                            ]),
                    ])
                    ->contained(true),
            ])
            ->columns(1);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getRecordSubNavigation(Page $page): array
    {
        return $page->generateNavigationItems([
            Pages\EditCarrier::class,
            Pages\ManageBankAccounts::class,
        ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListCarriers::route('/'),
            'create' => Pages\CreateCarrier::route('/create'),
            'edit' => Pages\EditCarrier::route('/{record}/edit'),
            'bank_accounts' => Pages\ManageBankAccounts::route('/{record}/bank-accounts'),
        ];
    }
}
