<?php

namespace App\Filament\Resources\VehicleResource\Widgets;

use Filament\Tables;
use Filament\Tables\Table;
use Filament\Widgets\TableWidget as BaseWidget;
use Filament\Support\Enums\FontWeight;

use Illuminate\Support\Carbon;

use App\Models\Certificate;
use App\Enums\VehicleCertificateType;

class ExpiredVehicleCertificates extends BaseWidget
{
    public function table(Table $table): Table
    {
        return $table
            ->heading('Araç Sertifikaları')
            ->description('Son 30 gün içerisinde geçerliliğini yitirecek sertifikalara gözatın.')
            ->query(
                Certificate::query()
                    ->where('model_type', 'App\Models\Vehicle')
                    ->whereBetween('ended_at', [Carbon::now(), Carbon::now()->addDays(30)])
                    ->with('certificateable')
            )
            ->columns([
                Tables\Columns\TextColumn::make('certificate_type_id')
                    ->label('Sertifika')
                    ->formatStateUsing(fn ($state): string => VehicleCertificateType::from($state)->getLabel())
                    ->color(fn ($state) => VehicleCertificateType::from($state)->getColor())
                    ->weight(FontWeight::SemiBold)
                    ->description(fn ($record) => $record->certificateable?->license_plate)
                    ,

                Tables\Columns\TextColumn::make('ended_at')
                    ->label('Geçerlilik Tarihi')
                    ->formatStateUsing(function ($record) {
                        $startedAt = Carbon::parse($record->started_at)->translatedFormat('j F Y');
                        $endedAt = Carbon::parse($record->ended_at)->translatedFormat('j F Y');

                        return '
                            <div class="grid grid-flow-col justify-stretch" style="width:280px;">
                                <div class="">'.$startedAt.'</div>
                                <div class="text-primary-500 dark:text-primary-400">- - --></div>
                                <div class="text-custom-600 dark:text-custom-400 font-medium" style="--c-400:var(--danger-400);--c-600:var(--danger-600);">
                                    '.$endedAt.'
                                </div>
                            </div>
                        ';
                    })
                    ->description(function ($state): string {
                        $expiryDate = Carbon::parse($state);
                        $today = Carbon::today();

                        return $expiryDate->isPast()
                            ? 'Geçerlilik süresi doldu.'
                            : 'Süresinin dolmasına '.$today->diff($expiryDate).' kaldı.';
                    })
                    ->html()
                    ->width('1%'),
            ])
            ->defaultSort('ended_at', 'asc')
            ->defaultPaginationPageOption(5)
            ->emptyStateHeading(__('filament-tables::table.empty.heading', ['model' => 'Süresi Dolan Sertifika']))
            ->emptyStateDescription('Geçerlilik süresinin son 30 günü kalan bir sertifika bulunmamaktadır.');
    }
}
