<?php

namespace App\Filament\Resources\VehicleResource\Pages;

use App\Filament\Resources\VehicleResource;
use App\Models\InsuranceCompany;

use App\Enums\VehicleCertificateType;

use Filament\Actions;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Resources\Pages\ManageRelatedRecords;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Enums\FiltersLayout;
use Filament\Infolists;
use Filament\Infolists\Infolist;
use Filament\Support\Enums\FontWeight;
use Filament\Support\Enums\MaxWidth;
use Filament\Support\RawJs;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Contracts\Support\Htmlable;
use Illuminate\Support\HtmlString;
use Illuminate\Support\Carbon;

use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use Filament\Infolists\Components\SpatieMediaLibraryImageEntry;

class ManageCertificates extends ManageRelatedRecords
{
    protected static string $resource = VehicleResource::class;

    protected static string $relationship = 'certificates';

    protected static ?string $navigationIcon = 'heroicon-s-document-text';
    protected static ?string $navigationLabel = 'Belgeler & Sertifikalar';

    private static ?string $modelLabel = 'Belge';

    public function getTitle(): string | Htmlable
    {
        return self::$navigationLabel;
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('certificate_type_id')
                    ->label('Belge Türü')
                    ->options(VehicleCertificateType::class)
                    ->required()
                    ->searchable()
                    ->native(false)
                    ->live()
                    ->afterStateUpdated(function (Forms\Components\Select $component) {
                        $component
                            ->getContainer()
                            ->getComponent('dynamicCertificateFields')
                            ->getChildComponentContainer()
                            ->fill();
                    })
                    ->disabled(fn ($operation) => $operation === 'edit')
                    ->columnSpanFull(),

                Forms\Components\DatePicker::make('started_at')
                    ->label('Başlangıç Tarihi')
                    ->required(),

                Forms\Components\DatePicker::make('ended_at')
                    ->label('Bitiş Tarihi')
                    ->required(),

                Forms\Components\Grid::make(1)
                    ->schema(fn (Get $get): array => VehicleResource::getCertificateFields($get('certificate_type_id')))
                    ->key('dynamicCertificateFields'),

                // SpatieMediaLibraryFileUpload::make('attachments123')
                //     ->label('Dosya Eki')
                //     ->disk('medias')
                //     ->conversion('preview')
                //     ->multiple()
                //     ->reorderable()
                //     ->appendFiles()
                //     ->downloadable()
                //     ->acceptedFileTypes([
                //         'image/jpeg',
                //         'image/jpeg',
                //         'image/png',
                //         'application/pdf',
                //         'application/vnd.ms-excel',
                //         'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                //         'application/msword',
                //         'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                //     ])
                //     // ->panelAspectRatio('1:1')
                //     ->removeUploadedFileButtonPosition('right')
                //     ->panelLayout('grid')
                //     ->columnSpanFull(),
        ])
        ->columns(2);
    }

    public function table(Table $table): Table
    {
        return $table
            // ->recordTitleAttribute('name')
            // ->recordTitle(fn ($record): string => {$record->name})
            // ->heading('Clients')
            // ->description('Manage your clients here.')
            // ->modelLabel(self::$modelLabel)
            ->defaultSort('id', 'desc')
            ->modifyQueryUsing(fn (Builder $query) => $query->with('insuranceCompany'))
            ->searchDebounce('1000ms')
            ->columns([
                Tables\Columns\TextColumn::make('certificate_type_id')
                    ->label('Sertifika')
                    ->formatStateUsing(fn ($state): string => VehicleCertificateType::from($state)->getLabel())
                    ->color(fn ($state) => VehicleCertificateType::from($state)->getColor())
                    ->weight(FontWeight::SemiBold)
                    ->description(fn ($record) => $record->insuranceCompany->name ?? '-'),

                Tables\Columns\TextColumn::make('policy_number')
                    ->label('Poliçe Numarası')
                    ->weight(FontWeight::Medium)
                    ->description(fn ($record) => $record->insurance_agent ?? '-')
                    ->default('-')
                    ->searchable(),

                Tables\Columns\TextColumn::make('ended_at')
                    ->label('Geçerlilik Tarihi')
                    // ->date('j F Y')
                    ->formatStateUsing(function ($record) {
                        $expiryDate = Carbon::parse($record->ended_at);
                        $today = Carbon::today();

                        $color = $expiryDate->isPast()
                            ? 'danger'
                            : 'success';

                        $startedAt = Carbon::parse($record->started_at)->translatedFormat('j F Y');
                        $endedAt = Carbon::parse($record->ended_at)->translatedFormat('j F Y');

                        return '
                            <div class="grid grid-flow-col justify-stretch" style="width:280px;">
                                <div class="">'.$startedAt.'</div>
                                <div class="text-primary-500 dark:text-primary-400">- - --></div>
                                <div class="text-custom-600 dark:text-custom-400 font-medium" style="--c-400:var(--'.$color.'-400);--c-600:var(--'.$color.'-600);">
                                    '.$endedAt.'
                                </div>
                            </div>
                        ';
                    })
                    ->description(function ($state): string {
                        $expiryDate = Carbon::parse($state);
                        $today = Carbon::today();

                        return $expiryDate->isPast()
                            ? 'Geçerlilik süresi doldu.'
                            : 'Süresinin dolmasına '.$today->diff($expiryDate).' kaldı.';
                    })
                    ->html()
                    ->width('1%'),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('Oluşturma')
                    ->dateTime()
                    ->toggleable(isToggledHiddenByDefault: true),

                Tables\Columns\TextColumn::make('updated_at')
                    ->label('Güncelleme')
                    ->dateTime()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('certificate_type_id')
                    ->label('Belge Türü')
                    ->options(VehicleCertificateType::class)
                    ->searchable(),

                Tables\Filters\SelectFilter::make('insurance_company_id')
                    ->label('Sigorta Şirketi')
                    ->options(InsuranceCompany::all()->pluck('name', 'id'))
                    ->searchable(),

                Tables\Filters\SelectFilter::make('ended_at')
                    ->label('Geçerlilik Tarihi')
                    ->options([
                        'expired'   => 'Süresi Dolanlar',
                        'valid'     => 'Geçerli Olanlar',
                    ])
                    ->query(function (Builder $query, array $data) {
                        match ($data['value']) {
                            'expired'   => $query->where('ended_at', '<', now()),
                            'valid'     => $query->where('ended_at', '>', now()),
                            default     => null,
                        };
                    })
                    ->native(false),

            ], layout: FiltersLayout::Modal)
            ->filtersTriggerAction(
                fn (\Filament\Tables\Actions\Action $action) => $action->slideOver()
            )
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->modelLabel(self::$modelLabel)
                    ->modalWidth(MaxWidth::TwoExtraLarge)
                    ->slideOver()
                    ->createAnother(false),
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    // Tables\Actions\ViewAction::make()
                    //     ->modalHeading(__('filament-actions::view.single.modal.heading', ['label' => self::$modelLabel]))
                    //     ->modalWidth(MaxWidth::Large)
                    //     ->slideOver(),

                    Tables\Actions\EditAction::make()
                        ->modalHeading(__('filament-actions::edit.single.modal.heading', ['label' => self::$modelLabel]))
                        ->modalWidth(MaxWidth::TwoExtraLarge)
                        ->slideOver(),

                    Tables\Actions\DeleteAction::make()
                        ->modalHeading(__('filament-actions::delete.single.modal.heading', ['label' => self::$modelLabel])),
                ]),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->emptyStateHeading(__('filament-tables::table.empty.heading', ['model' => self::$modelLabel]))
            ->emptyStateDescription(__('filament-tables::table.empty.description', ['model' => self::$modelLabel]));
    }
}
